#include "pico.h"

static struct vtor_rec __attribute__((aligned(256))) vtor = { 0 };

static void
xosc_init(void)
{
	xosc.ctrl = 0x00fabaa0;		/* ENABLE and 1_15MHZ */

	/* default XOSC_STARTUP is 4.2ms at 12MHz...datasheet suggests 1ms is
	 * enough. no need to change it. */

	while (!(xosc.status & 0x80000000));	/* wait for XOSC STABLE */

	clocks.ref_ctrl = 2;		/* clk_ref from XOSC */
	/* clk_sys etc will follow clk_ref */

	rosc.ctrl = 0x00d1e000;		/* disable ROSC */

/* XXX - to enable USB, set CLK_USB_CTRL to 0x00000840 ENABLE|clksrc_pll_usb,
after bringing up that PLL at 4x */
}

static void
wdt_init(void)
{
	wdt.load = 0x00ffffff;
	wdt.tick = 0x20c;	/* enable, cycles=12, 1us off 12MHz XOSC?? */

	/* 0x0001fff8 (reset everything but ROSC/XOSC/CLOCKS) works too but
	 * let's go for the heavier reboot */
	psm.wdsel = 0x0001ffff;			/* WDT resets everything */
	HW_OR(wdt.ctrl, 0x40000000);		/* enable */
}

static void
vtor_init(void)
{
	scb.vtor = &vtor;
}

/* reboot by enabling the WDT with a 50us countdown and then infinite looping
 * until the WDT hits 0 */
static void
reboot(void)
{
	psm.wdsel = 0x0001ffff;	/* WDT resets everything */
	wdt.tick = 0x20c;	/* enable, cycles=12, 1us off 12MHz XOSC?? */
	wdt.load = 100;
	while (1) {
		__asm("");
	}
}

/* Use the systick timer to count out a delay in ms */
/* NB - rvr/cvr are only 24-bit so this is capped at 1398ms at 12MHz */
static void
delay(unsigned ms)
{
	syst.rvr = (12000000/1000)*ms;
	syst.cvr = 0;			/* reset systick */
	syst.csr = 0x00000005;		/* clksource=clk_sys, no interrupt, yes
					 * enabled */
	while (!(syst.csr & 0x00010000));	/* wait for COUNTFLAG */
}

/* register the callback to be called by the systick timer every so-many-ms (up
 * to 1398ms at 12MHz) */
static void
systick_register(unsigned ms, void (*cb)(void))
{
	vtor.systick = cb;

	syst.rvr = (12000000/1000)*ms;
	syst.cvr = 0;			/* reset systick */
	syst.csr = 0x00000007;		/* clksource=clk_sys, yes interrupt,
					 * yes enabled */
}

/* enables the LED drive, then led_toggle() to turn it on or off */
static void
led_init(void)
{
	HW_ANDNOT(resets.reset, 1<<5);	/* enable IO_BANK0 */
	/* OEOVER=3 enable output, OUTOVER=1 output low, FUNCSEL=31 (NULL) */
	io_bank0.gpio[25].ctrl = 0x0000321f;
}

static void
led_toggle(void)
{
	HW_XOR(io_bank0.gpio[25].ctrl,  0x00000100);
}


void
main(void)
{
	xosc_init();
	wdt_init();
	vtor_init();

	led_init();
	systick_register(333, led_toggle);
	while (1) {
		__asm("wfi");
	}
}
